package ga.core.algorithm.automatic;

import ga.core.GA;
import ga.core.evaluation.IFitnessEvaluator;
import ga.core.goperators.ICrossoverOp;
import ga.core.goperators.IMutationOp;
import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IClusterPopulation;
import ga.core.individual.population.IPopulation;
import ga.core.selection.ISelector;
import ga.core.validation.GAContext;
import ga.core.validation.IValidator;

import java.util.logging.Logger;

/**
 * Generational simple genetic algorithm.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class SGAGeneration<T extends IIndividual<T>> implements GA<T> {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(SGAGeneration.class
      .getName());

  private static final int CRITICAL_WHILE_ITERATIONS = 100000;

  private final IPopulation<T> population;
  private final ISelector<T> selector;
  private final IMutationOp<T> mutateOp;
  private final ICrossoverOp<T> crossoverOp;
  private IValidator<T> validator;
  private final GAContext gaContext;

  private boolean validate;
  private boolean useEliteStrategy;

  private int generation;

  /**
   * Creates a new generational SGA.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * @param context
   *          The ga context.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SGAGeneration(final IPopulation<T> population,
      final IFitnessEvaluator<T> evaluator, final ISelector<T> selector,
      final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final boolean useEliteStrategy,
      final GAContext context) {
    this.population = population;
    this.selector = selector;
    this.mutateOp = mutateOperator;
    this.crossoverOp = crossoverOperator;
    this.gaContext = context;
    this.useEliteStrategy = useEliteStrategy;
    population.setEvaluator(evaluator);
  }

  /**
   * Creates a new generational SGA. The required ga context is newly created.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SGAGeneration(final IPopulation<T> population,
      final IFitnessEvaluator<T> evaluator, final ISelector<T> selector,
      final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final boolean useEliteStrategy) {
    this(population, evaluator, selector, mutateOperator, crossoverOperator,
        useEliteStrategy, new GAContext());
  }

  @SuppressWarnings("rawtypes")
  @Override
  public void init() {
    population.initRandomly(validate ? validator : null, gaContext);

    if (population instanceof IClusterPopulation) {
      ((IClusterPopulation) population).doClustering();
    }

    population.evaluateAutomatic();
  }

  @SuppressWarnings("rawtypes")
  @Override
  public void step() {
    final IndividualList<T> nextGeneration = new IndividualList<T>();

    // SELECT
    while (nextGeneration.size() < population.size()) {
      // fill the next generation
      final IndividualList<T> newIndividuals = doReproduction();

      for (final T newIndividual : newIndividuals) {
        if (population.isAllowDuplicates()
            || !nextGeneration.contains(newIndividual)) {
          nextGeneration.add(newIndividual);
        }
      }

      while (nextGeneration.size() > population.size()) {
        // if it has exeeded the limit, remove elements until it fits the size
        nextGeneration.remove(0);
      }
    }

    if (useEliteStrategy) {
      nextGeneration.set(0, getPopulation().getEliteIndividual().clone());
    }

    // REPLACE OLD GENERATION
    generation++;
    population.clear();
    population.addIndividuals(nextGeneration);

    if (population instanceof IClusterPopulation) {
      ((IClusterPopulation) population).doClustering();
    }

    // EVALUATE
    population.evaluateAutomatic();
  }

  /**
   * Selects two individuals, recombines and mutates them.
   * 
   * @return A list of two offsprings.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private IndividualList<T> doReproduction() {
    IndividualList<T> selectedIndividuals = selector.select(population);

    final T ind1 = selectedIndividuals.get(0);
    final T ind2 = selectedIndividuals.get(1);

    int whileCounter = 0;

    do {
      // CROSSOVER
      final IndividualList<T> newSelectedIndividuals = crossoverOp.crossover(
          ind1, ind2, gaContext);
      if (newSelectedIndividuals != null) {
        selectedIndividuals = newSelectedIndividuals;

      } else {
        LOGGER.warning("Crossover operation returned null");
      }

      // since we do not know how much individuals are returned by the
      // crossover, mutate all
      for (int i = 0; i < selectedIndividuals.size(); i++) {
        final T ind = selectedIndividuals.get(i);
        // MUTATE
        final T mutatedIndividual = mutateOp.mutate(ind, gaContext);
        if (mutatedIndividual != null) {
          selectedIndividuals.set(i, mutatedIndividual);

        } else {
          LOGGER.warning("Mutation operation returned null");
        }
      }

      whileCounter++;

      if (whileCounter > CRITICAL_WHILE_ITERATIONS) {
        LOGGER
            .warning("Critical iterations exceeded. Endless loop? Individuals: "
                + ind1 + "   " + ind2);

        return selectedIndividuals;
      }

    } while (validate && validator != null
        && !selectedIndividuals.isValid(validator, gaContext));

    return selectedIndividuals;
  }

  @Override
  public void setValidate(final boolean validate) {
    if (validate && validator == null) {
      throw new RuntimeException("Error - no validator has been set");
    }

    this.validate = validate;
  }

  @Override
  public void setValidator(final IValidator<T> validator) {
    this.validator = validator;

    if (validator != null) {
      validate = true;
    }
  }

  @Override
  public IPopulation<T> getPopulation() {
    return population;
  }
}
